/*
 * MIT License
 *
 * Copyright (c) 2002-2025 Mikko Tommila
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package org.apfloat.jscience;

import javolution.text.Text;
import javolution.xml.XMLFormat.InputElement;
import javolution.xml.XMLFormat.OutputElement;
import javolution.xml.stream.XMLStreamException;

import org.apfloat.Apcomplex;
import org.apfloat.ApcomplexMath;
import org.apfloat.Apfloat;
import org.apfloat.ApfloatContext;
import org.apfloat.ApfloatMath;
import org.apfloat.Apint;
import org.jscience.mathematics.number.Number;
import org.jscience.mathematics.structure.Field;

/**
 * This class represents an arbitrary precision field object.
 *
 * @param <T> The type of the field.
 * @param <V> The type of the underlying value.
 *
 * @since 1.8.0
 * @version 1.15.0
 * @author Mikko Tommila
 */

public abstract class AbstractField<T extends AbstractField<T, V>, V extends Apcomplex>
    extends Number<T>
    implements Field<T>
{
    /**
     * Constructs a new field object with the specified value.
     *
     * @param value The value.
     */

    protected AbstractField(V value)
    {
        if (value == null)
        {
            throw new NullPointerException("Value can't be null");
        }
        this.value = value;
    }

    /**
     * Returns the sum of this object with the one specified.
     *
     * @param that The addend.
     *
     * @return <code>this + that</code>
     */

    @Override
    public abstract T plus(T that);

    /**
     * Returns the additive inverse of this object.
     *
     * @return <code>-this</code>
     */

    @Override
    public abstract T opposite();

    /**
     * Returns the product of this object with the one specified.
     *
     * @param that The multiplicand.
     *
     * @return <code>this * that</code>
     */

    @Override
    public abstract T times(T that);


    /**
     * Returns the multiplicative inverse of this object.
     *
     * @return <code>1 / this</code>
     *
     * @exception ArithmeticException If the divisor is zero.
     */

    @Override
    public abstract T inverse()
        throws ArithmeticException;

    /**
     * Returns a copy of this object.
     *
     * @return A copy of this object.
     */

    @Override
    public abstract T copy();

    /**
     * Compares the absolute value of this number
     * with the absolute value of the number specified.
     *
     * @param that The number to be compared with.
     *
     * @return <code>|this| &gt; |that|</code>
     */

    @Override
    public boolean isLargerThan(T that)
    {
        return ApcomplexMath.norm(value()).compareTo(ApcomplexMath.norm(that.value())) > 0;
    }

    /**
     * Returns the value of this number as the underlying type.
     *
     * @return The value.
     */

    public V value()
    {
        return this.value;
    }

    /**
     * Returns the value of this number as a <code>double</code>.
     *
     * @return The value.
     */

    @Override
    public double doubleValue()
    {
        return value().doubleValue();
    }

    /**
     * Returns the value of this number as a <code>long</code>.
     *
     * @return The value.
     */

    @Override
    public long longValue()
    {
        return value().longValue();
    }

    /**
     * Returns the text representation of this number.
     *
     * @return The string representation of this number as a <code>Text</code>.
     */

    @Override
    public Text toText()
    {
        return Text.valueOf(value().toString());
    }

    /**
     * Compares this number to another number.
     *
     * @param that The number to be compared with.
     *
     * @return -1, 0, or 1 depending on the ordering. 
     */

    @Override
    public int compareTo(T that)
    {
        int result = value().real().compareTo(that.value().real());
        if (result == 0)
        {
            result = value().imag().compareTo(that.value().imag());
        }
        return result;
    }

    /**
     * Returns the hash code for this number.
     *
     * @return The hash code value.
     */

    @Override
    public int hashCode()
    {
        return value().hashCode();
    }

    /**
     * Compares for equality.
     * 
     * @return If the objects are equal.
     */

    @Override
    public boolean equals(Object obj)
    {
        if (obj instanceof AbstractField)
        {
            AbstractField<?, ?> that = (AbstractField<?, ?>) obj;
            return value().equals(that.value());
        }
        return false;
    }

    static Apfloat parse(String prefix, InputElement xml)
        throws XMLStreamException
    {
        String mantissa = xml.getAttribute(prefix + "mantissa", "0");
        long exponent = Long.parseLong(xml.getAttribute(prefix + "exponent", "0")),
             precision = Long.parseLong(xml.getAttribute(prefix + "precision", String.valueOf(mantissa.length())));
        int radix = Integer.parseInt(xml.getAttribute(prefix + "radix", String.valueOf(ApfloatContext.getContext().getDefaultRadix())));
        return ApfloatMath.scale(new Apint(mantissa, radix).precision(precision), exponent);
    }

    static void format(Apfloat value, String prefix, OutputElement xml)
        throws XMLStreamException
    {
        format(value, prefix, xml, value.precision());
    }

    static void format(Apfloat value, String prefix, OutputElement xml, Long precision)
        throws XMLStreamException
    {
        Apint mantissa = ApfloatMath.scale(value, Math.subtractExact(value.size(), value.scale())).truncate();
        long exponent = (value.signum() == 0 ? 0 : Math.subtractExact(value.scale(), value.size()));
        xml.setAttribute(prefix + "mantissa", mantissa.toString());
        xml.setAttribute(prefix + "exponent", exponent);
        if (precision != null)
        {
            xml.setAttribute(prefix + "precision", precision.longValue());
        }
        xml.setAttribute(prefix + "radix", value.radix());
    }

    private static final long serialVersionUID = -7725271295007354895L;

    private V value;
}
